

#define SVC_CLIENT_IMPL		// This is GroveOS application

#include <stdio.h>
#include <string.h>
#include "msg.h"
#include "svc.h"
#include "tfs.h"
#include "modbus_common.h"

#define	LEDPAD_WIDTH		24	
#define	LEDPAD_HEIGHT		9	
#define	NUM_OF_LEDS		(LEDPAD_WIDTH * LEDPAD_HEIGHT)
#define LED_LEV0  		(28)	// PWM for '0'
#define LED_LEV1  		(62)	// PWM for '1'
#define	DISPLAY_WORK		0
#define	DISPLAY_RED		1
#define	DISPLAY_GREEN		2
#define	DISPLAY_BLUE		3


#define	RAINBOW_MSG_MODBUS_READ_HOLD_REGS	2001
#define	RAINBOW_MSG_MODBUS_WRITE_HOLD_REGS	2002
#define	RAINBOW_MSG_DISPLAY_RED			2003
#define	RAINBOW_MSG_DISPLAY_GREEN		2004
#define	RAINBOW_MSG_DISPLAY_BLUE		2005
#define	RAINBOW_MSG_DISPLAY_WORK		2006

#define	REG_VERSION		1024	// { Device, OS, Application }
#define	REG_LED_PRINT		1028
#define	REG_LED_BLIT		1030
#define	REG_LED_CLEAR		1032
#define	REG_COLOR		1034
#define	REG_MAX_COLOR		1036
#define	REG_X			1038
#define	REG_Y			1040
#define	REG_SCROLL_X		1042
#define	REG_SCROLL_Y		1044
#define	REG_MODE		1046
#define	REG_BITMAP_WIDTH	1048
#define	REG_BITMAP_HEIGHT	1050
#define	REG_FLIP_X		1052
#define	REG_FLIP_Y		1054
#define	REG_FONT_WIDTH		1056
#define	REG_FONT_HEIGHT		1058
#define	REG_FONT_NAME		1060


#define	DEFAULT_DATA		"\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac\xac ***  1 *** " 
#define	DEFAULT_DATA_SIZE	3078		// reserve some space in backup memory for user's data
#define	DEFAULT_DATA_X		LEDPAD_WIDTH	
#define	DEFAULT_DATA_Y		0
#define	DEFAULT_MAX_COLOR	0x00000300
#define	DEFAULT_COLOR		0x00ffffff
#define	DEFAULT_SCROLL_X	2		// 100 / 4 = 25Hz (25 pixels/sec)	
#define	DEFAULT_SCROLL_Y	0
#define	DEFAULT_MODE		0		// 0 - data mode, 1 - bitmap mode
#define	DEFAULT_BITMAP_WIDTH	0
#define	DEFAULT_BITMAP_HEIGHT	0
#define	DEFAULT_FLIP_X		0		// 0 - no flipping, 1 - flipped 
#define	DEFAULT_FLIP_Y		0		// 0 - no flipping, 1 - flipped 
#define	DEFAULT_FONT_WIDTH	6		// pixels	
#define	DEFAULT_FONT_HEIGHT	8		// pixels
#define	DEFAULT_FONT_NAME	""	
#define	MODE_DATA		0		// in data mode use REG_LED_PRINT to load displaying data
#define	MODE_BITMAP		1		// in bitmap mode use REG_LED_BLIT to load displaying image 

#define	VAR_CHECK		"CHECK"
#define	VAR_MODE		"MODE"
#define	VAR_DATA		"DATA"
#define	VAR_DATA_X		"DATA_X"
#define	VAR_DATA_Y		"DATA_Y"
#define	VAR_MAX_COLOR		"MAX_COLOR"
#define	VAR_COLOR		"COLOR"
#define	VAR_SCROLL_X		"SCROLL_X"
#define	VAR_SCROLL_Y		"SCROLL_Y"
#define	VAR_BITMAP_WIDTH	"BITMAP_W"
#define	VAR_BITMAP_HEIGHT	"BITMAP_H"
#define	VAR_FLIP_X		"FLIP_X"
#define	VAR_FLIP_Y		"FLIP_Y"
#define	VAR_FONT_WIDTH		"FONT_W"
#define	VAR_FONT_HEIGHT		"FONT_H"
#define	VAR_FONT_NAME		"FONT_NAME"


int application_text_addr; // required by LIBC

char str[128]; // some buffer for debug messages

uint16_t* bitstream_buffer = NULL;	// DMA transfer buffer
uint32_t rgb_buf[NUM_OF_LEDS];		// RGB buffer to store image in RGB format
extern const char font_6x8[];
uint8_t *font = NULL;
uint32_t *data_x = NULL;
uint32_t *data_y = NULL;
uint32_t *max_color = NULL;
uint32_t *color = NULL;
int32_t *scroll_x = NULL;
int32_t *scroll_y = NULL;
int32_t *mode = NULL;
int32_t *bitmap_w = NULL;
int32_t *bitmap_h = NULL;
int32_t *flip_x = NULL;
int32_t *flip_y = NULL;
int32_t *font_width = NULL;
int32_t *font_height = NULL;
char *data = NULL;
char *font_name = NULL;
int pos_x, pos_y;
int test_state;

int baud_rates[] = { 9600, 19200, 38400, 57600, 115200, 250000 };
char *stop_bits[] = { "1", "0.5", "2" , "1.5" };
char parity_list[] = { 'N', 'E', 'O' };

MODBUS_RESPONSE* modbus_response_write = NULL;
MODBUS_RESPONSE* modbus_response_read = NULL;
uint16_t version[3] = { DEVICE_VERSION, 0, RAINBOW_VERSION };


#define swap16(u16)     ((uint16_t)(((uint16_t)(u16) >> 8) | ((uint16_t)(u16) << 8)))
#define	SATURATE8(X)	(X > 255 ? 255 : X < 0 ? 0: X)
#define	ABS(X)		((X) > 0 ? (X) : (-1 * (X)))
#define MIN(X,Y)	((X) < (Y) ? (X) : (Y))


uint32_t GiveRainbowColor(double position)
{
  /* if position > 1 then we have repetition of colors it maybe useful    */
      
	if (position>1.0){
		if (position-(int)position==0.0)
			position=1.0; 
		else 
			position=position-(int)position;
	}
  
  
 
  
  unsigned char nmax=6; /* number of color segments */
  double m=nmax* position;
  
  int n=(int)m; // integer of m
  
  double f=m-n;  // fraction of m
  unsigned char t=(int)(f*255);

	uint8_t r,g,b;

  
switch( n){
   case 0: {
      r = 255;
      g = t;
      b = 0;
       break;
    };
   case 1: {
      r = 255 - t;
      g = 255;
      b = 0;
       break;
    };
   case 2: {
      r = 0;
      g = 255;
      b = t;
       break;
    };
   case 3: {
      r = 0;
      g = 255 - t;
      b = 255;
       break;
    };
   case 4: {
      r = t;
      g = 0;
      b = 255;
       break;
    };
   case 5: {
      r = 255;
      g = 0;
      b = 255 - t;
       break;
    };
    default: {
      r = 255;
      g = 0;
      b = 0;
       break;
    };

}; // case

	return (r << 16) | (g << 8) | (b);
}

void rainbow_print_char(char c, int x, int y)
{
	int my_y;
	int _color = x * 16;

/*
	uint32_t r = (*color >> 16) & 0xff;
	uint32_t g = (*color >> 8) & 0xff;
	uint32_t b = (*color >> 0 ) & 0xff;

	uint32_t rgb_sum = r + g + b;

	if(rgb_sum > *max_color) {
		r = r * *max_color / rgb_sum;
		g = g * *max_color / rgb_sum;
		b = b * *max_color / rgb_sum;
	}
*/
	
	int lines = *font_height / 8;

	if(*font_height % 8) 
		lines++;

	for(int line = 0; line < lines; line++) {

		int line_height = MIN(8, (*font_height - line * 8));
		int my_x = x;

		for(int col = 0; col < *font_width; col++) {

			if(my_x >= 0 && my_x < LEDPAD_WIDTH) {


				char bit_mask = 0x01;
				uint32_t f;
			
				f = font[*font_width * c * lines + line * *font_width + col];

				my_y = y + line * 8;

				for(int bit = 0; bit < line_height; bit++) {

					if(my_y >= 0 && my_y < LEDPAD_HEIGHT) {

						uint32_t _color = GiveRainbowColor((ABS(my_x - pos_x + my_y) % 256) / 256.0);

						uint32_t r = (_color >> 16) & 0xff;
						uint32_t g = (_color >> 8) & 0xff;
						uint32_t b = (_color >> 0 ) & 0xff;

						uint32_t rgb_sum = r + g + b;

						if(rgb_sum > *max_color) {
							r = r * *max_color / rgb_sum;
							g = g * *max_color / rgb_sum;
							b = b * *max_color / rgb_sum;
						}

						int pixel_idx = my_y * LEDPAD_WIDTH + my_x; 

					
						if(f & bit_mask) {
							rgb_buf[pixel_idx] = (r << 16) | (g << 8) | b;
						} else {
							rgb_buf[pixel_idx] = 0;
						}
					}

					bit_mask = bit_mask << 1;
					my_y++;
				}
			}

			my_x++;
		}
	}
}


void rainbow_print(int size, char *buf, int x, int y)
{
	//svc_debug_print(str, sprintf(str, "RAINBOW rainbow_print: buf = %p, x = %d, y = %d, size = %d\r\n", buf, x, y, size));

        for(int i = 0; i < size; i++) {
                rainbow_print_char(*buf++, x, y); // x - is a back reference
                x += *font_width;
        }

}


void rainbow_blit(uint16_t* buf, int w, int h, int x, int y)
{

	uint16_t* p = buf;

	for(int i = 0; i < h; i++) {
		int d_y = y + i;
		if(d_y >= 0 && d_y < LEDPAD_HEIGHT) {
			for(int j = 0; j < w; j++) {
				int d_x = x+j;
				if(d_x >= 0 && d_x < LEDPAD_WIDTH) {
					//svc_debug_print(str, sprintf(str, "XXX: i = %d, j = %d, p = %p\r\n", i, j ,p));
					uint16_t rgb565 = swap16(*p);
					uint32_t r = ((rgb565 >> 11) & 0x1f) << 3;
					uint32_t g = ((rgb565 >> 5) & 0x3f) << 2;
					uint32_t b = ((rgb565 >> 0) & 0x1f) << 3;
					uint32_t rgb_sum = r + g + b;

					if(rgb_sum > *max_color) {
						r = r * *max_color / rgb_sum;
						g = g * *max_color / rgb_sum;
						b = b * *max_color / rgb_sum;
					}
	
					rgb_buf[d_y*LEDPAD_WIDTH+d_x] = (r << 16) | (g << 8) | (b << 0);
				}
				p++;
				if((char*)p - (char*)buf >= DEFAULT_DATA_SIZE) 
					return;
			}
		} else {
			buf += w; // skip invisible line of pixels
		}
	}
}



void rainbow_clear(void)
{
	memset(rgb_buf, 0, sizeof(*rgb_buf) * NUM_OF_LEDS);
}


void rainbow_render(void)
{
	
	uint32_t rgb;
	uint16_t *p = bitstream_buffer + 24;

	// Convert RGB to PWM bit stream
	for(int y = 0 ; y < LEDPAD_HEIGHT; y++) {
		for(int x = 0; x < LEDPAD_WIDTH; x++) {
			uint32_t X, Y;

			if(*flip_x) 
				X = LEDPAD_WIDTH - x - 1;
			else
				X = x;

			if(*flip_y) 
				Y = LEDPAD_HEIGHT - y - 1;
			else
				Y = y;
			
			rgb = rgb_buf[X + Y*LEDPAD_WIDTH];
			for(int i = 0; i < 8; i++) {
				// Green
				if ( ((rgb >> 8) >> (7 - i)) & 1 ) *(p+0) = LED_LEV1;
				else *(p+0) = LED_LEV0;
				// Red
				if ( ((rgb >> 16) >> (7 - i)) & 1 ) *(p+8) = LED_LEV1;
				else *(p+8) = LED_LEV0;
				// Blue
				if ( ((rgb >> 0) >> (7 - i)) & 1 ) *(p+16) = LED_LEV1;
				else *(p+16) = LED_LEV0;
				p++;
			}
			p += 16; // step over R and B 
		}
	}

}


uint32_t event_counter = 0;

// Once installed this procedure will be called for every system event, like timers, IRQ sources, ADC, etc...
void process_queue(MSG *msg)
{
        switch(msg->message) {

                case INFO_TIMER_INT: { // msg->p1 contains millisecond timer counter value

			uint32_t info_timer_ms = msg->p1;

			event_counter++;

			if(test_state != DISPLAY_WORK ) {
				if(event_counter % 50 == 0) {
					for(int i = 0; i < LEDPAD_WIDTH * LEDPAD_HEIGHT; i++) {
						switch(test_state) {
							case DISPLAY_RED:
								rgb_buf[i] = 0x00ff0000;
								break;
							case DISPLAY_GREEN:
								rgb_buf[i] = 0x0000ff00;
								break;
							case DISPLAY_BLUE:
								rgb_buf[i] = 0x000000ff;
								break;
							default:
								rgb_buf[i] = 0x00000000;
								break;
						}
					}
				}
				rainbow_render();
				svc_dc_pwm_transfer(DC_PORT_1, (void*)bitstream_buffer, (NUM_OF_LEDS + 2) * 24, 0); 
				break;
			}

			if(info_timer_ms % 1000 == 0) {

				svc_debug_print(str, sprintf(str, "RAINBOW is running, pos_x = %d, pos_y = %d, strlen = %d, color = %p, scroll_x = %d, scroll_y = %d,  mode = %s, bitmap = (%dx%d), flip_x = %d, flip_y = %d\r\n", pos_x, pos_y, strlen(data), *color, *scroll_x, *scroll_y,  *mode ? "BITMAP":"TEXT", *bitmap_w, *bitmap_h, *flip_x, *flip_y));

			}


			if(*scroll_x == 0 && *scroll_y == 0) {
				if(event_counter % 50 == 0) {
					rainbow_clear();
					if(*mode == MODE_BITMAP) 
						rainbow_blit((uint16_t*)data, *bitmap_w, *bitmap_h, *data_x, *data_y);
					else
						rainbow_print(strlen(data), data, *data_x, *data_y);
				}
			}

			if(*scroll_x > 0) {
				if(event_counter % *scroll_x == 0) {
					int min_pos_x;
			
					rainbow_clear();

					if(*mode == MODE_BITMAP) { 
						min_pos_x = -1 * *bitmap_w;
						rainbow_blit((uint16_t*)data, *bitmap_w, *bitmap_h, pos_x, pos_y);
					} else {
						rainbow_print(strlen(data), data, pos_x, pos_y);
						min_pos_x = -1 * strlen(data) * *font_width;
					}

					pos_x--;
					if(pos_x < min_pos_x) {
						pos_x = *data_x;
					}
				}
			}

			if(*scroll_x < 0) {
				if(event_counter % ABS(*scroll_x) == 0) {
					int max_pos_x;

					rainbow_clear();

					if(*mode == MODE_BITMAP) {
						rainbow_blit((uint16_t*)data, *bitmap_w, *bitmap_h, pos_x, pos_y);
						max_pos_x = LEDPAD_WIDTH;
					} else {
						rainbow_print(strlen(data), data, pos_x, pos_y);
						max_pos_x = LEDPAD_WIDTH;
					}

					pos_x++;
					if(pos_x > max_pos_x) {
						pos_x = -1 * strlen(data) * *font_width;
					}
				}
			}

			if(*scroll_y > 0) {
				if(event_counter % *scroll_y == 0) {
					int min_pos_y;

					rainbow_clear();

					if(*mode == MODE_BITMAP) {
						rainbow_blit((uint16_t*)data, *bitmap_w, *bitmap_h, pos_x, pos_y);
						min_pos_y = -1 * *bitmap_h;
					} else {
						rainbow_print(strlen(data), data, pos_x, pos_y);
						min_pos_y = -1 * *font_height;
					}

					pos_y--;
					if(pos_y < min_pos_y) {
						pos_y = LEDPAD_HEIGHT;
					}
				}
			}

			if(*scroll_y < 0) {
				if(info_timer_ms % ABS(*scroll_y) == 0) {
					int max_pos_y;

					rainbow_clear();

					if(*mode == MODE_BITMAP) { 
						rainbow_blit((uint16_t*)data, *bitmap_w, *bitmap_h, pos_x, pos_y);
						max_pos_y = LEDPAD_HEIGHT;
					} else {
						rainbow_print(strlen(data), data, pos_x, pos_y);
						max_pos_y = LEDPAD_HEIGHT;
					}

					pos_y++;
					if(pos_y > max_pos_y) {
						pos_y = -1 * *font_height;
					}
				}
			}


			if(info_timer_ms % 20 == 0) {
				rainbow_render();
				svc_dc_pwm_transfer(DC_PORT_1, (void*)bitstream_buffer, (NUM_OF_LEDS + 2) * 24, 0); 
			}

		} break;

		
		case RAINBOW_MSG_DISPLAY_RED: {
			test_state = DISPLAY_RED;
			svc_softtimer_run(RAINBOW_MSG_DISPLAY_GREEN, 2000, 0, 0);
			svc_debug_print(str, sprintf(str, "Displaying RED screen\r\n"));
		} break;

		case RAINBOW_MSG_DISPLAY_GREEN: {
			test_state = DISPLAY_GREEN;
			svc_softtimer_run(RAINBOW_MSG_DISPLAY_BLUE, 2000, 0, 0);
			svc_debug_print(str, sprintf(str, "Displaying GREEN screen\r\n"));
		} break;

		case RAINBOW_MSG_DISPLAY_BLUE: {
			test_state = DISPLAY_BLUE;
			svc_softtimer_run(RAINBOW_MSG_DISPLAY_WORK, 2000, 0, 0);
			svc_debug_print(str, sprintf(str, "Displaying BLUE screen\r\n"));
		} break;

		case RAINBOW_MSG_DISPLAY_WORK: {
			test_state = DISPLAY_WORK;
			svc_debug_print(str, sprintf(str, "Test finished, working...\r\n"));
		} break;

		case DC_PWM_1_DMA_TC: {
			//svc_debug_print(str, sprintf(str, "DC_PWM_1_DMA_TC"));
			//rainbow_render();
			//svc_dc_pwm_transfer(DC_PORT_1, (void*)bitstream_buffer, (NUM_OF_LEDS + 2) * 24, 0); 
		} break;
		

		
		case RAINBOW_MSG_MODBUS_READ_HOLD_REGS: {

				MODBUS_RESPONSE *resp = (MODBUS_RESPONSE *)msg->p1;
				
				svc_debug_print(str, sprintf(str, "Received READ request over Modbus: reg = %d, qty = %d\r\n", MODBUS_GET_REQ_REG(resp), MODBUS_GET_REQ_QTY(resp)));

				MODBUS_RESPONSE_START(resp, MODBUS_GET_REQ_FUNC(resp));
				
				if(MODBUS_GET_REQ_REG(resp) >= REG_VERSION && (MODBUS_GET_REQ_REG(resp) + MODBUS_GET_REQ_QTY(resp) - 1 < REG_VERSION+3)) {
				
					for(int i = MODBUS_GET_REQ_REG(resp); i < MODBUS_GET_REQ_REG(resp) + MODBUS_GET_REQ_QTY(resp); i++) {
						MODBUS_RESPONSE_ADD_WORD(resp, version[i-1024]);
					}
				} else {
					MODBUS_RESPONSE_ERROR(resp, FUNC_READ_HOLD_REGS, ERR_WRONG_ARGS);
				}
 
				svc_modbus1_submit_response(resp);
		} break;

		case RAINBOW_MSG_MODBUS_WRITE_HOLD_REGS: {

				MODBUS_RESPONSE *resp = (MODBUS_RESPONSE *)msg->p1;

				svc_debug_print(str, sprintf(str, "Received WRITE HOLD REGS request over Modbus: reg = %d, qty = %d\r\n", MODBUS_GET_REQ_REG(resp), MODBUS_GET_REQ_QTY(resp)));

				MODBUS_RESPONSE_START(resp, MODBUS_GET_REQ_FUNC(resp));
				
				switch(MODBUS_GET_REQ_REG(resp)) {
					case REG_LED_PRINT: {
						int size_in_bytes = resp->rxbuf[6];
						char *buf = resp->rxbuf+7+4;
						int x = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						int y = (resp->rxbuf[9] << 8) | resp->rxbuf[10];
						svc_vault_set(VAR_DATA, (void *)buf, MIN(size_in_bytes, DEFAULT_DATA_SIZE), 0);
						svc_vault_set(VAR_DATA_X, (void *)&x, sizeof(x), 0);
						svc_vault_set(VAR_DATA_Y, (void *)&y, sizeof(y), 0);
						MODBUS_RESPONSE_OK(resp, MODBUS_GET_REQ_FUNC(resp));
					} break;

					case REG_LED_BLIT: {
						int size_in_bytes = resp->rxbuf[6] - 4;
						char *buf = resp->rxbuf+7+4;
						int offset = (resp->rxbuf[9] << 8) | resp->rxbuf[10];

						svc_debug_print(str, sprintf(str, "BLIT: size = %d, offset = %d\r\n", size_in_bytes, offset));
		
						if(offset + size_in_bytes > DEFAULT_DATA_SIZE ) {
							MODBUS_RESPONSE_ERROR(resp, FUNC_READ_HOLD_REGS, ERR_WRONG_ARGS);
							svc_debug_print(str, sprintf(str, "BLIT: wrong params, size = %d, offset = %d\r\n", size_in_bytes, offset));
						} else {
							int rc = svc_vault_set(VAR_DATA, (void *)buf, MIN(size_in_bytes, DEFAULT_DATA_SIZE-offset), offset);
							//svc_debug_print(str, sprintf(str, "BLIT: rc = %d\r\n", rc));
							MODBUS_RESPONSE_OK(resp, MODBUS_GET_REQ_FUNC(resp));
						}
					} break;

					case REG_FONT_NAME: {
						int size_in_bytes = resp->rxbuf[6];
						char *font_name = resp->rxbuf+7;
						font_name[size_in_bytes] = 0;
						svc_vault_set(VAR_FONT_NAME, (void *)font_name, MIN(size_in_bytes+1, 16), 0);
						TFS_HEADER* font_tfs = svc_tfs_find(font_name);
						if(font_tfs) {
							font = ((char*)font_tfs) + sizeof(TFS_HEADER);
							svc_debug_print(str, sprintf(str, "Using font: %s\r\n", font_name));
							MODBUS_RESPONSE_OK(resp, MODBUS_GET_REQ_FUNC(resp));
						} else {
							svc_debug_print(str, sprintf(str, "Font file not found: %s\r\n", font_name));
							MODBUS_RESPONSE_ERROR(resp, FUNC_READ_HOLD_REGS, ERR_WRONG_ARGS);
						}
					} break;


					case REG_LED_CLEAR: {
						rainbow_clear();
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
					} break;

					case REG_COLOR: {
						int col_h = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						int col_l = (resp->rxbuf[9] << 8) | resp->rxbuf[10];
						int col = (col_h << 16 ) | col_l;
						svc_vault_set(VAR_COLOR, (void *)&col, sizeof(col), 0);
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
					} break;

					case REG_MAX_COLOR: {
						int max_col_h = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						int max_col_l = (resp->rxbuf[9] << 8) | resp->rxbuf[10];
						int max_col = (max_col_h << 16 ) | max_col_l;
						svc_vault_set(VAR_MAX_COLOR, (void *)&max_col, sizeof(max_col), 0);
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
					} break;

					case REG_X: {
						int x_h = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						int x_l = (resp->rxbuf[9] << 8) | resp->rxbuf[10];
						int x = (x_h << 16 ) | x_l;
						svc_vault_set(VAR_DATA_X, (void *)&x, sizeof(x), 0);
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
						pos_x = *data_x;
					} break;

					case REG_Y: {
						int y_h = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						int y_l = (resp->rxbuf[9] << 8) | resp->rxbuf[10];
						int y = (y_h << 16 ) | y_l;
						svc_vault_set(VAR_DATA_Y, (void *)&y, sizeof(y), 0);
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
						pos_y = *data_y;
					} break;

					case REG_SCROLL_X: {
						int scroll_x_h = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						int scroll_x_l = (resp->rxbuf[9] << 8) | resp->rxbuf[10];
						int scroll_x = (scroll_x_h << 16 ) | scroll_x_l;
						svc_vault_set(VAR_SCROLL_X, (void *)&scroll_x, sizeof(scroll_x), 0);
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
					} break;

					case REG_SCROLL_Y: {
						int scroll_y_h = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						int scroll_y_l = (resp->rxbuf[9] << 8) | resp->rxbuf[10];
						int scroll_y = (scroll_y_h << 16 ) | scroll_y_l;
						svc_vault_set(VAR_SCROLL_Y, (void *)&scroll_y, sizeof(scroll_y), 0);
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
					} break;

					case REG_MODE: {
						int mode_h = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						int mode_l = (resp->rxbuf[9] << 8) | resp->rxbuf[10];
						int mode = (mode_h << 16 ) | mode_l;
						svc_vault_set(VAR_MODE, (void *)&mode, sizeof(mode), 0);
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
					} break;

					case REG_BITMAP_WIDTH: {
						int bitmap_w_h = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						int bitmap_w_l = (resp->rxbuf[9] << 8) | resp->rxbuf[10];
						int bitmap_w = (bitmap_w_h << 16 ) | bitmap_w_l;
						svc_vault_set(VAR_BITMAP_WIDTH, (void *)&bitmap_w, sizeof(bitmap_w), 0);
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
					} break;

					case REG_BITMAP_HEIGHT: {
						int bitmap_h_h = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						int bitmap_h_l = (resp->rxbuf[9] << 8) | resp->rxbuf[10];
						int bitmap_h = (bitmap_h_h << 16 ) | bitmap_h_l;
						svc_vault_set(VAR_BITMAP_HEIGHT, (void *)&bitmap_h, sizeof(bitmap_h), 0);
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
					} break;

					case REG_FLIP_X: {
						int flip_x = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						svc_vault_set(VAR_FLIP_X, (void *)&flip_x, sizeof(flip_x), 0);
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
					} break;

					case REG_FLIP_Y: {
						int flip_y = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						svc_vault_set(VAR_FLIP_Y, (void *)&flip_y, sizeof(flip_y), 0);
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
					} break;

					case REG_FONT_WIDTH: {
						int f_w = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						svc_vault_set(VAR_FONT_WIDTH, (void *)&f_w, sizeof(f_w), 0);
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
					} break;

					case REG_FONT_HEIGHT: {
						int f_h = (resp->rxbuf[7] << 8) | resp->rxbuf[8];
						svc_vault_set(VAR_FONT_HEIGHT, (void *)&f_h, sizeof(f_h), 0);
						MODBUS_RESPONSE_OK(resp, FUNC_READ_HOLD_REGS);
					} break;

					default:
						MODBUS_RESPONSE_ERROR(resp, FUNC_READ_HOLD_REGS, ERR_WRONG_ARGS);
						break;
				}

				svc_modbus1_submit_response(resp);

		} break;

		default:
			break; // just skip all other system events
	}
}


void* _start(void)
{

	int rc;


        application_text_addr = svc_get_text(); // Request GroveOS to provide our data segment address, this is needed by LIBC to work properly 

	version[1] = svc_get_os_version();

	svc_debug_print(str, sprintf(str, "Rainbow LED Display. Copyright (C) Fabmicro, LLC., Tyumen, Russia, 2017. Version: %02X.%d.%d\r\n", version[0], version[1], version[2]));

	// Load config variables
	svc_debug_print(str, sprintf(str, "Loading config variables from backup memory.\r\n"));

	uint32_t *check;
	uint32_t need_defaults = 0;

	if(svc_vault_get(VAR_CHECK, (void **)&check) < 0) {
		need_defaults = 1;
	} else {
		if(*check != 0x01ABCDEF) 
			need_defaults = 1;
	}

	if(need_defaults) {
		svc_debug_print(str, sprintf(str, "Backup memory corrupted, loading default values.\r\n"));
		char *data = (char*) rgb_buf; // use it as temporarily long buffer
		int addr, baud, stop, parity;
		svc_get_config(CONFIG_EXT1_MODBUS_ADDR, &addr);
		svc_get_config(CONFIG_EXT1_BAUD_RATE, &baud);
		svc_get_config(CONFIG_EXT1_STOP_BITS, &stop);
		svc_get_config(CONFIG_EXT1_PARITY, &parity);
		snprintf(data, 1024, DEFAULT_DATA, addr, baud_rates[MIN(baud, 6)], parity_list[MIN(parity, 2)], stop_bits[MIN(stop, 4)]);
		svc_vault_set(VAR_DATA, (void *)data, DEFAULT_DATA_SIZE, 0);
		uint32_t def_color = DEFAULT_COLOR;
		svc_vault_set(VAR_COLOR, (void *)&def_color, sizeof(def_color), 0);
		uint32_t def_max_color = DEFAULT_MAX_COLOR;
		svc_vault_set(VAR_MAX_COLOR, (void *)&def_max_color, sizeof(def_max_color), 0);
		uint32_t def_x = DEFAULT_DATA_X;
		svc_vault_set(VAR_DATA_X, (void *)&def_x, sizeof(def_x), 0);
		uint32_t def_y = DEFAULT_DATA_Y;
		svc_vault_set(VAR_DATA_Y, (void *)&def_y, sizeof(def_y), 0);
		uint32_t def_scroll_x = DEFAULT_SCROLL_X;
		svc_vault_set(VAR_SCROLL_X, (void *)&def_scroll_x, sizeof(def_scroll_x), 0);
		uint32_t def_scroll_y = DEFAULT_SCROLL_Y;
		svc_vault_set(VAR_SCROLL_Y, (void *)&def_scroll_y, sizeof(def_scroll_y), 0);
		uint32_t check = 0x01ABCDEF;
		svc_vault_set(VAR_CHECK, (void *)&check, sizeof(check), 0);
		uint32_t def_mode = DEFAULT_MODE;
		svc_vault_set(VAR_MODE, (void *)&def_mode, sizeof(def_mode), 0);
		uint32_t def_bitmap_width = DEFAULT_BITMAP_WIDTH;
		svc_vault_set(VAR_BITMAP_WIDTH, (void *)&def_bitmap_width, sizeof(def_bitmap_width), 0);
		uint32_t def_bitmap_height = DEFAULT_BITMAP_HEIGHT;
		svc_vault_set(VAR_BITMAP_HEIGHT, (void *)&def_bitmap_height, sizeof(def_bitmap_height), 0);
		uint32_t def_flip_x = DEFAULT_FLIP_X;
		svc_vault_set(VAR_FLIP_X, (void *)&def_flip_x, sizeof(def_flip_x), 0);
		uint32_t def_flip_y = DEFAULT_FLIP_Y;
		svc_vault_set(VAR_FLIP_Y, (void *)&def_flip_y, sizeof(def_flip_y), 0);
		uint32_t def_font_w = DEFAULT_FONT_WIDTH;
		svc_vault_set(VAR_FONT_WIDTH, (void *)&def_font_w, sizeof(def_font_w), 0);
		uint32_t def_font_h = DEFAULT_FONT_HEIGHT;
		svc_vault_set(VAR_FONT_HEIGHT, (void *)&def_font_h, sizeof(def_font_h), 0);
		svc_vault_set(VAR_FONT_NAME, (void *)DEFAULT_FONT_NAME, 16, 0);
	}

	if(svc_vault_get(VAR_COLOR, (void **)&color) < 0) {
		uint32_t def_color = DEFAULT_COLOR;
		svc_vault_set(VAR_COLOR, (void *)&def_color, sizeof(def_color), 0);
		svc_vault_get(VAR_COLOR, (void **)&color);
	}

	if(svc_vault_get(VAR_MAX_COLOR, (void **)&max_color) < 0) {
		uint32_t def_max_color = DEFAULT_MAX_COLOR;
		svc_vault_set(VAR_MAX_COLOR, (void *)&def_max_color, sizeof(def_max_color), 0);
		svc_vault_get(VAR_MAX_COLOR, (void **)&max_color);
	}

	if(svc_vault_get(VAR_DATA_X, (void **)&data_x) < 0) {
		uint32_t def_x = DEFAULT_DATA_X;
		svc_vault_set(VAR_DATA_X, (void *)&def_x, sizeof(def_x), 0);
		svc_vault_get(VAR_DATA_X, (void **)&data_x);
	}

	if(svc_vault_get(VAR_DATA_Y, (void **)&data_y) < 0) {
		uint32_t def_y = DEFAULT_DATA_Y;
		svc_vault_set(VAR_DATA_Y, (void *)&def_y, sizeof(def_y), 0);
		svc_vault_get(VAR_DATA_Y, (void **)&data_y);
	}

	if(svc_vault_get(VAR_DATA, (void **)&data) < 0) {
		char *data = (char*) rgb_buf; // use it as temporarily long buffer
		int addr, baud, stop, parity;
		svc_get_config(CONFIG_EXT1_MODBUS_ADDR, &addr);
		svc_get_config(CONFIG_EXT1_BAUD_RATE, &baud);
		svc_get_config(CONFIG_EXT1_STOP_BITS, &stop);
		svc_get_config(CONFIG_EXT1_PARITY, &parity);
		snprintf(data, 1024, DEFAULT_DATA, addr, baud_rates[MIN(baud, 6)], parity_list[MIN(parity, 2)], stop_bits[MIN(stop, 4)]);
		svc_vault_set(VAR_DATA, (void *)data, DEFAULT_DATA_SIZE, 0);
		svc_vault_get(VAR_DATA, (void **)&data);
	}

	if(svc_vault_get(VAR_SCROLL_X, (void **)&scroll_x) < 0) {
		uint32_t def_scroll_x = DEFAULT_SCROLL_X;
		svc_vault_set(VAR_SCROLL_X, (void *)&def_scroll_x, sizeof(def_scroll_x), 0);
		svc_vault_get(VAR_SCROLL_X, (void **)&scroll_x);
	}

	if(svc_vault_get(VAR_SCROLL_Y, (void **)&scroll_y) < 0) {
		uint32_t def_scroll_y = DEFAULT_SCROLL_Y;
		svc_vault_set(VAR_SCROLL_Y, (void *)&def_scroll_y, sizeof(def_scroll_y), 0);
		svc_vault_get(VAR_SCROLL_Y, (void **)&scroll_y);
	}

	if(svc_vault_get(VAR_MODE, (void **)&mode) < 0) {
		uint32_t def_mode = DEFAULT_MODE;
		svc_vault_set(VAR_MODE, (void *)&def_mode, sizeof(def_mode), 0);
		svc_vault_get(VAR_MODE, (void **)&mode);
	}

	if(svc_vault_get(VAR_BITMAP_WIDTH, (void **)&bitmap_w) < 0) {
		uint32_t def_bitmap_w = DEFAULT_BITMAP_WIDTH;
		svc_vault_set(VAR_BITMAP_WIDTH, (void *)&def_bitmap_w, sizeof(def_bitmap_w), 0);
		svc_vault_get(VAR_BITMAP_WIDTH, (void **)&bitmap_w);
	}

	if(svc_vault_get(VAR_BITMAP_HEIGHT, (void **)&bitmap_h) < 0) {
		uint32_t def_bitmap_h = DEFAULT_BITMAP_HEIGHT;
		svc_vault_set(VAR_BITMAP_HEIGHT, (void *)&def_bitmap_h, sizeof(def_bitmap_h), 0);
		svc_vault_get(VAR_BITMAP_HEIGHT, (void **)&bitmap_h);
	}

	if(svc_vault_get(VAR_FLIP_X, (void **)&flip_x) < 0) {
		uint32_t def_flip_x = DEFAULT_FLIP_X;
		svc_vault_set(VAR_FLIP_X, (void *)&def_flip_x, sizeof(def_flip_x), 0);
		svc_vault_get(VAR_FLIP_X, (void **)&flip_x);
	}

	if(svc_vault_get(VAR_FLIP_Y, (void **)&flip_y) < 0) {
		uint32_t def_flip_y = DEFAULT_FLIP_Y;
		svc_vault_set(VAR_FLIP_Y, (void *)&def_flip_y, sizeof(def_flip_y), 0);
		svc_vault_get(VAR_FLIP_Y, (void **)&flip_y);
	}

	if(svc_vault_get(VAR_FONT_WIDTH, (void **)&font_width) < 0) {
		uint32_t def_font_w = DEFAULT_FONT_WIDTH;
		svc_vault_set(VAR_FONT_WIDTH, (void *)&def_font_w, sizeof(def_font_w), 0);
		svc_vault_get(VAR_FONT_WIDTH, (void **)&font_width);
	}

	if(svc_vault_get(VAR_FONT_HEIGHT, (void **)&font_height) < 0) {
		uint32_t def_font_h = DEFAULT_FONT_HEIGHT;
		svc_vault_set(VAR_FONT_HEIGHT, (void *)&def_font_h, sizeof(def_font_h), 0);
		svc_vault_get(VAR_FONT_HEIGHT, (void **)&font_height);
	}

	if(svc_vault_get(VAR_FONT_NAME, (void **)&font_name) < 0) {
		svc_vault_set(VAR_FONT_NAME, (void *)DEFAULT_FONT_NAME, 16, 0);
		svc_vault_get(VAR_FONT_NAME, (void **)&font_name);
	}

	svc_debug_print(str, sprintf(str, "Loaded pointers: color = %p, max_color = %p, data_x = %p, data_y = %p, data = %p, scroll_x = %p, scroll_y = %p\r\n",
			color, max_color, data_x, data_y, data, scroll_x, scroll_y));

	svc_debug_print(str, sprintf(str, "Loaded pointers: mode = %p, bitmap_w = %p, bitmap_h = %p, font_name = %p, font_w = %p, font_h = %p\r\n", 
			mode, bitmap_w, bitmap_h, font_name, font_width, font_height));

	if(color == NULL || max_color == NULL || data_x == NULL || data_y == NULL || data ==  NULL || scroll_x == NULL || scroll_x == NULL || scroll_y == NULL || 
	   mode == NULL || bitmap_w == NULL || bitmap_h == NULL || flip_x == NULL || flip_y == NULL || font_width == NULL || font_height == NULL || font_name == NULL) {

		svc_debug_print(str, sprintf(str, "One or more vault variables have not been laoded, execution terminated!\r\n")); 
		return (void*)-1;
	}

//	svc_debug_print(str, sprintf(str, "Loaded variables: color = %p, max_color = %p, data_x = %d, data_y = %d, data = %s, scroll_x = %d, scroll_y = %d, mode = %s, bitmap = (%dx%d)\r\n", 
//			*color, *max_color, *data_x, *data_y, data, *scroll_x, *scroll_y, *mode ? "BITMAP":"TEXT", *bitmap_w, *bitmap_h));


	TFS_HEADER* font_tfs = svc_tfs_find(font_name);
	if(font_tfs) {
		font = ((char*)font_tfs) + sizeof(TFS_HEADER);
		svc_debug_print(str, sprintf(str, "Using font: %s (%dx%d)\r\n", font_name, *font_width, *font_height));
	} else {
		font = (uint8_t*)font_6x8;
		int def_font_w = DEFAULT_FONT_WIDTH;
		svc_vault_set(VAR_FONT_WIDTH, (void *)&def_font_w, sizeof(def_font_w), 0);
		int def_font_h = DEFAULT_FONT_HEIGHT;
		svc_vault_set(VAR_FONT_HEIGHT, (void *)&def_font_h, sizeof(def_font_h), 0);
		svc_debug_print(str, sprintf(str, "Using default font (%dx%d)\r\n", *font_width, *font_height));
	}


	int mem_size = (NUM_OF_LEDS + 2) * 24 * sizeof(*bitstream_buffer);
	svc_debug_print(str, sprintf(str, "Allocating %d bytes for bitstream\r\n", mem_size));

	bitstream_buffer = (uint16_t*) svc_malloc(mem_size);

	if(!bitstream_buffer) {
		svc_debug_print(str, sprintf(str, "Got memory allocation error\r\n"));
		return (void*)-1;
	}

	memset((void*)bitstream_buffer, 0, (NUM_OF_LEDS + 2) * 24 * sizeof(*bitstream_buffer));
	memset(rgb_buf, 0, NUM_OF_LEDS * 4);


	pos_x = *data_x;
	pos_y = *data_y;


	// Init Modbus stuff

	mem_size = sizeof(MODBUS_RESPONSE);
	svc_debug_print(str, sprintf(str, "Allocating %d bytes for modbus REG responder\r\n", mem_size));
	modbus_response_write = (MODBUS_RESPONSE*) svc_malloc(mem_size);
	if(!modbus_response_write) {
		svc_debug_print(str, sprintf(str, "Got memory allocation error\r\n"));
		return (void*)-1;
	}

	MODBUS_CREATE_RESPONDER(modbus_response_write, RAINBOW_MSG_MODBUS_WRITE_HOLD_REGS, FUNC_WRITE_MANY_HOLD_REGS, 1024, 65535);

	if((rc = svc_modbus1_register_responder(modbus_response_write)) < 0) {
		svc_debug_print(str, sprintf(str, "Failed to register Modbus responder, rc = %d\r\n", rc));
		return (void*)-1;
	}


	svc_debug_print(str, sprintf(str, "Allocating %d bytes for modbus ID responder\r\n", mem_size));
	modbus_response_read = (MODBUS_RESPONSE*) svc_malloc(mem_size);
	if(!modbus_response_read) {
		svc_debug_print(str, sprintf(str, "Got memory allocation error\r\n"));
		return (void*)-1;
	}

	MODBUS_CREATE_RESPONDER(modbus_response_read, RAINBOW_MSG_MODBUS_READ_HOLD_REGS, FUNC_READ_HOLD_REGS, 1024, 65535);

	if((rc = svc_modbus1_register_responder(modbus_response_read)) < 0) {
		svc_debug_print(str, sprintf(str, "Failed to register Modbus responder, rc = %d\r\n", rc));
		return (void*)-1;
	}

	svc_debug_print(str, sprintf(str, "Modbus responder has been registered\r\n"));
	
	rainbow_render();

	// Init WS2812B stuff

	svc_set_dc_clock(800000); // T=1.25 us
	svc_set_dc_pwm(DC_PORT_1, 0); // init PWM mode 
	// Initiate DMA transfer, +2 cycles at the end to form a RESET signal
	//svc_dc_pwm_transfer(DC_PORT_1, (void*)bitstream_buffer, (NUM_OF_LEDS + 2) * 24, 0); 

	test_state = -1;
	svc_softtimer_run(RAINBOW_MSG_DISPLAY_RED, 1, 0, 0);

	return &process_queue; // return address of our queue processing procedure
}



